#' @title 3D Surface Plot for QQ Regression
#'
#' @description
#' Creates an interactive 3D surface plot of QQ regression results using 
#' plotly. Supports multiple color scales including MATLAB-style Jet.
#'
#' @param qq_result An object of class "qq_regression" or a data frame 
#'   with columns: y_quantile, x_quantile, coefficient, r_squared, p_value.
#' @param type Character string. Type of values to plot: "coefficient" 
#'   (default), "rsquared", or "pvalue".
#' @param colorscale Character string. Color scale to use. Options are:
#'   \itemize{
#'     \item "Jet" - MATLAB-style rainbow (default)
#'     \item "BlueRed" - Blue to red diverging scale
#'     \item "Viridis" - Viridis perceptually uniform scale
#'     \item "Plasma" - Plasma perceptually uniform scale
#'   }
#' @param show_contour Logical. If \code{TRUE}, shows gridlines on the 
#'   surface. Default is \code{TRUE}.
#' @param x_label Character string. Label for x-axis. Default is 
#'   "X Quantile (tau)".
#' @param y_label Character string. Label for y-axis. Default is 
#'   "Y Quantile (theta)".
#' @param title Character string. Plot title. If \code{NULL}, an 
#'   automatic title is generated.
#' @param aspect_ratio Numeric vector of length 3. Aspect ratio for 
#'   x, y, z axes. Default is c(1, 1, 0.7).
#' @param camera List. Camera position settings for plotly. Default 
#'   provides a good viewing angle.
#'
#' @return A plotly object representing the 3D surface plot.
#'
#' @details
#' The function creates an interactive 3D surface plot where:
#' \itemize{
#'   \item X-axis: Quantiles of the independent variable (tau)
#'   \item Y-axis: Quantiles of the dependent variable (theta)
#'   \item Z-axis: Estimated coefficients, R-squared, or p-values
#' }
#'
#' The Jet colorscale mimics MATLAB's default color scheme for surface 
#' plots, transitioning from blue through cyan, green, yellow to red.
#'
#' @examples
#' # Generate example data
#' set.seed(42)
#' n <- 200
#' x <- rnorm(n)
#' y <- 0.5 * x + rnorm(n, sd = 0.5)
#'
#' # Run QQ regression
#' result <- qq_regression(y, x, verbose = FALSE)
#' 
#' \donttest{
#' # Create 3D surface plot of coefficients
#' plot_qq_3d(result, type = "coefficient", colorscale = "Jet")
#' 
#' # Create plot of R-squared values
#' plot_qq_3d(result, type = "rsquared", colorscale = "Viridis")
#' }
#'
#' @export
#' @importFrom plotly plot_ly layout "%>%"
plot_qq_3d <- function(qq_result, 
                       type = c("coefficient", "rsquared", "pvalue"),
                       colorscale = c("Jet", "BlueRed", "Viridis", "Plasma"),
                       show_contour = TRUE,
                       x_label = "X Quantile (tau)",
                       y_label = "Y Quantile (theta)",
                       title = NULL,
                       aspect_ratio = c(1, 1, 0.7),
                       camera = list(eye = list(x = 1.4, y = 1.7, z = 1.2))) {
  
  # Extract results
  if (inherits(qq_result, "qq_regression")) {
    results <- qq_result$results
  } else if (is.data.frame(qq_result)) {
    results <- qq_result
  } else {
    stop("'qq_result' must be a qq_regression object or data frame")
  }
  
  # Match arguments
  type <- match.arg(type)
  colorscale <- match.arg(colorscale)
  
  # Remove incomplete cases
  results <- results[complete.cases(results), ]
  
  if (nrow(results) == 0) {
    stop("No complete results to plot")
  }
  
  # Select plot variable
  plot_var <- switch(type,
                     "coefficient" = results$coefficient,
                     "rsquared" = results$r_squared,
                     "pvalue" = results$p_value)
  
  plot_title_suffix <- switch(type,
                              "coefficient" = "Coefficients",
                              "rsquared" = "R-squared",
                              "pvalue" = "P-values")
  
  if (is.null(title)) {
    title <- paste("3D Surface Plot -", plot_title_suffix)
  }
  
  # Get unique quantiles
  y_unique <- sort(unique(results$y_quantile))
  x_unique <- sort(unique(results$x_quantile))
  
  # Create z matrix
  z_matrix <- matrix(NA_real_, nrow = length(y_unique), ncol = length(x_unique))
  for (i in seq_along(y_unique)) {
    for (j in seq_along(x_unique)) {
      idx <- which(results$y_quantile == y_unique[i] & 
                     results$x_quantile == x_unique[j])
      if (length(idx) > 0) {
        z_matrix[i, j] <- plot_var[idx[1]]
      }
    }
  }
  
  # Calculate step sizes for contours
  x_step <- if (length(x_unique) > 1) {
    unique(round(diff(x_unique), 6))[1]
  } else {
    0.05
  }
  y_step <- if (length(y_unique) > 1) {
    unique(round(diff(y_unique), 6))[1]
  } else {
    0.05
  }
  
  # Create the plot
  p <- plot_ly(
    x = x_unique, 
    y = y_unique, 
    z = z_matrix,
    type = "surface",
    colorscale = colorscale,
    showscale = TRUE,
    colorbar = list(title = plot_title_suffix, tickformat = ".2f"),
    contours = list(
      x = list(show = show_contour, color = "black",
               start = min(x_unique), end = max(x_unique), size = x_step),
      y = list(show = show_contour, color = "black",
               start = min(y_unique), end = max(y_unique), size = y_step),
      z = list(show = FALSE)
    ),
    lighting = list(ambient = 0.55, diffuse = 0.8, 
                    specular = 0.15, roughness = 0.9),
    lightposition = list(x = 60, y = 120, z = 80),
    hovertemplate = paste0(
      "X Quantile: %{x:.2f}<br>",
      "Y Quantile: %{y:.2f}<br>",
      plot_title_suffix, ": %{z:.4f}<extra></extra>"
    )
  )
  
  p <- p %>% layout(
    title = title,
    paper_bgcolor = "white",
    plot_bgcolor = "white",
    margin = list(l = 10, r = 10, b = 10, t = 50),
    scene = list(
      xaxis = list(title = x_label, tickformat = ".2f",
                   showgrid = TRUE, gridcolor = "#E6E6E6",
                   zerolinecolor = "#D0D0D0", linecolor = "#C0C0C0"),
      yaxis = list(title = y_label, tickformat = ".2f",
                   showgrid = TRUE, gridcolor = "#E6E6E6",
                   zerolinecolor = "#D0D0D0", linecolor = "#C0C0C0"),
      zaxis = list(title = plot_title_suffix, showgrid = TRUE, 
                   gridcolor = "#F0F0F0"),
      aspectratio = list(x = aspect_ratio[1], y = aspect_ratio[2], 
                         z = aspect_ratio[3]),
      camera = camera
    )
  )
  
  return(p)
}


#' @title Heatmap for QQ Regression
#'
#' @description
#' Creates an interactive heatmap of QQ regression results using plotly.
#'
#' @param qq_result An object of class "qq_regression" or a data frame 
#'   with columns: y_quantile, x_quantile, coefficient, r_squared, p_value.
#' @param type Character string. Type of values to plot: "coefficient" 
#'   (default), "rsquared", or "pvalue".
#' @param colorscale Character string. Color scale to use. Options are:
#'   \itemize{
#'     \item "Viridis" - Viridis scale (default for coefficient)
#'     \item "Plasma" - Plasma scale (default for rsquared)
#'     \item "Jet" - MATLAB-style rainbow (default for pvalue)
#'     \item "BlueRed" - Blue to red diverging scale
#'   }
#' @param x_label Character string. Label for x-axis. Default is 
#'   "X Variable Quantiles".
#' @param y_label Character string. Label for y-axis. Default is 
#'   "Y Variable Quantiles".
#' @param title Character string. Plot title. If \code{NULL}, an 
#'   automatic title is generated.
#' @param zmin Numeric. Minimum value for color scale. If \code{NULL}, 
#'   automatically determined.
#' @param zmax Numeric. Maximum value for color scale. If \code{NULL}, 
#'   automatically determined.
#'
#' @return A plotly object representing the heatmap.
#'
#' @examples
#' # Generate example data
#' set.seed(42)
#' n <- 200
#' x <- rnorm(n)
#' y <- 0.5 * x + rnorm(n, sd = 0.5)
#'
#' # Run QQ regression
#' result <- qq_regression(y, x, verbose = FALSE)
#' 
#' \donttest{
#' # Create coefficient heatmap
#' plot_qq_heatmap(result, type = "coefficient")
#' 
#' # Create R-squared heatmap
#' plot_qq_heatmap(result, type = "rsquared", colorscale = "Plasma")
#' 
#' # Create p-value heatmap
#' plot_qq_heatmap(result, type = "pvalue", colorscale = "Jet")
#' }
#'
#' @export
#' @importFrom plotly plot_ly layout "%>%"
plot_qq_heatmap <- function(qq_result,
                            type = c("coefficient", "rsquared", "pvalue"),
                            colorscale = NULL,
                            x_label = "X Variable Quantiles",
                            y_label = "Y Variable Quantiles",
                            title = NULL,
                            zmin = NULL,
                            zmax = NULL) {
  
  # Extract results
  if (inherits(qq_result, "qq_regression")) {
    results <- qq_result$results
  } else if (is.data.frame(qq_result)) {
    results <- qq_result
  } else {
    stop("'qq_result' must be a qq_regression object or data frame")
  }
  
  # Match arguments
  type <- match.arg(type)
  
  # Remove incomplete cases
  results <- results[complete.cases(results), ]
  
  if (nrow(results) == 0) {
    stop("No complete results to plot")
  }
  
  # Select plot variable and set defaults
  plot_var <- switch(type,
                     "coefficient" = results$coefficient,
                     "rsquared" = results$r_squared,
                     "pvalue" = results$p_value)
  
  plot_title_suffix <- switch(type,
                              "coefficient" = "Coefficient Heatmap",
                              "rsquared" = "R-squared Heatmap",
                              "pvalue" = "P-values Heatmap")
  
  if (is.null(colorscale)) {
    colorscale <- switch(type,
                         "coefficient" = "Viridis",
                         "rsquared" = "Plasma",
                         "pvalue" = "Jet")
  }
  
  if (type == "pvalue") {
    if (is.null(zmin)) zmin <- 0
    if (is.null(zmax)) zmax <- 1
  }
  
  if (is.null(title)) {
    title <- plot_title_suffix
  }
  
  # Get unique quantiles
  y_unique <- sort(unique(results$y_quantile))
  x_unique <- sort(unique(results$x_quantile))
  
  # Create z matrix
  z_matrix <- matrix(NA_real_, nrow = length(y_unique), ncol = length(x_unique))
  for (i in seq_along(y_unique)) {
    for (j in seq_along(x_unique)) {
      idx <- which(results$y_quantile == y_unique[i] & 
                     results$x_quantile == x_unique[j])
      if (length(idx) > 0) {
        z_matrix[i, j] <- plot_var[idx[1]]
      }
    }
  }
  
  # Create the plot
  p <- plot_ly(
    x = x_unique,
    y = y_unique,
    z = z_matrix,
    type = "heatmap",
    colorscale = colorscale,
    showscale = TRUE,
    zmin = zmin,
    zmax = zmax,
    hovertemplate = paste0(
      "X Quantile: %{x:.2f}<br>",
      "Y Quantile: %{y:.2f}<br>",
      "Value: %{z:.4f}<extra></extra>"
    )
  )
  
  p <- p %>% layout(
    title = title,
    xaxis = list(title = x_label, tickfont = list(size = 10)),
    yaxis = list(title = y_label, tickfont = list(size = 10))
  )
  
  return(p)
}


#' @title Contour Plot for QQ Regression
#'
#' @description
#' Creates an interactive contour plot of QQ regression coefficients using 
#' plotly.
#'
#' @param qq_result An object of class "qq_regression" or a data frame 
#'   with columns: y_quantile, x_quantile, coefficient.
#' @param type Character string. Type of values to plot: "coefficient" 
#'   (default), "rsquared", or "pvalue".
#' @param colorscale Character string. Color scale to use. Default is "Jet".
#' @param show_labels Logical. If \code{TRUE}, shows contour labels. 
#'   Default is \code{TRUE}.
#' @param x_label Character string. Label for x-axis. Default is 
#'   "X Variable Quantiles".
#' @param y_label Character string. Label for y-axis. Default is 
#'   "Y Variable Quantiles".
#' @param title Character string. Plot title. If \code{NULL}, an 
#'   automatic title is generated.
#'
#' @return A plotly object representing the contour plot.
#'
#' @examples
#' # Generate example data
#' set.seed(42)
#' n <- 200
#' x <- rnorm(n)
#' y <- 0.5 * x + rnorm(n, sd = 0.5)
#'
#' # Run QQ regression
#' result <- qq_regression(y, x, verbose = FALSE)
#' 
#' \donttest{
#' # Create contour plot
#' plot_qq_contour(result)
#' }
#'
#' @export
#' @importFrom plotly plot_ly layout "%>%"
plot_qq_contour <- function(qq_result,
                            type = c("coefficient", "rsquared", "pvalue"),
                            colorscale = "Jet",
                            show_labels = TRUE,
                            x_label = "X Variable Quantiles",
                            y_label = "Y Variable Quantiles",
                            title = NULL) {
  
  # Extract results
  if (inherits(qq_result, "qq_regression")) {
    results <- qq_result$results
  } else if (is.data.frame(qq_result)) {
    results <- qq_result
  } else {
    stop("'qq_result' must be a qq_regression object or data frame")
  }
  
  # Match arguments
  type <- match.arg(type)
  
  # Remove incomplete cases
  results <- results[complete.cases(results), ]
  
  if (nrow(results) == 0) {
    stop("No complete results to plot")
  }
  
  # Select plot variable
  plot_var <- switch(type,
                     "coefficient" = results$coefficient,
                     "rsquared" = results$r_squared,
                     "pvalue" = results$p_value)
  
  plot_title_suffix <- switch(type,
                              "coefficient" = "Coefficient",
                              "rsquared" = "R-squared",
                              "pvalue" = "P-value")
  
  if (is.null(title)) {
    title <- paste(plot_title_suffix, "Contour Plot")
  }
  
  # Create the plot
  p <- plot_ly(
    x = results$x_quantile,
    y = results$y_quantile,
    z = plot_var,
    type = "contour",
    colorscale = colorscale,
    showscale = TRUE,
    contours = list(showlabels = show_labels)
  )
  
  p <- p %>% layout(
    title = title,
    xaxis = list(title = x_label),
    yaxis = list(title = y_label)
  )
  
  return(p)
}


#' @title Quantile Correlation Heatmap
#'
#' @description
#' Creates an interactive heatmap showing the correlation between quantiles 
#' of two variables. Uses a blue-red diverging color scale similar to 
#' Python's seaborn style.
#'
#' @param y Numeric vector. The first variable.
#' @param x Numeric vector. The second variable.
#' @param quantiles Numeric vector. Quantiles to compute correlations for. 
#'   Default is \code{seq(0.1, 0.9, by = 0.1)}.
#' @param x_label Character string. Label for x-axis. Default is 
#'   "X Variable Quantiles".
#' @param y_label Character string. Label for y-axis. Default is 
#'   "Y Variable Quantiles".
#' @param title Character string. Plot title. Default is 
#'   "Quantile Correlation Heatmap".
#' @param show_annotations Logical. If \code{TRUE}, shows correlation 
#'   values as text annotations. Default is \code{TRUE}.
#'
#' @return A plotly object representing the correlation heatmap.
#'
#' @details
#' This function computes correlations between binary indicators of whether 
#' observations fall below each quantile threshold. The resulting heatmap 
#' shows how the relationship between variables varies across different 
#' parts of their distributions.
#'
#' @examples
#' # Generate example data
#' set.seed(42)
#' n <- 200
#' x <- rnorm(n)
#' y <- 0.5 * x + rnorm(n, sd = 0.5)
#' 
#' \donttest{
#' # Create correlation heatmap
#' plot_qq_correlation(y, x)
#' }
#'
#' @export
#' @importFrom plotly plot_ly layout "%>%"
#' @importFrom stats quantile cor complete.cases
plot_qq_correlation <- function(y, x,
                                 quantiles = seq(0.1, 0.9, by = 0.1),
                                 x_label = "X Variable Quantiles",
                                 y_label = "Y Variable Quantiles",
                                 title = "Quantile Correlation Heatmap",
                                 show_annotations = TRUE) {
  
  # Input validation
  if (!is.numeric(y) || !is.numeric(x)) {
    stop("'y' and 'x' must be numeric vectors")
  }
  
  if (length(y) != length(x)) {
    stop("'y' and 'x' must have the same length")
  }
  
  # Handle missing values
  complete_idx <- complete.cases(y, x)
  y_data <- y[complete_idx]
  x_data <- x[complete_idx]
  
  # Compute correlation matrix
  corr_matrix <- matrix(NA_real_, nrow = length(quantiles), 
                        ncol = length(quantiles))
  rownames(corr_matrix) <- paste0("Y_Q", quantiles)
  colnames(corr_matrix) <- paste0("X_Q", quantiles)
  
  for (i in seq_along(quantiles)) {
    for (j in seq_along(quantiles)) {
      y_quant <- quantile(y_data, quantiles[i], na.rm = TRUE)
      x_quant <- quantile(x_data, quantiles[j], na.rm = TRUE)
      y_binary <- as.numeric(y_data <= y_quant)
      x_binary <- as.numeric(x_data <= x_quant)
      corr_matrix[i, j] <- cor(y_binary, x_binary, use = "complete.obs")
    }
  }
  
  # Create the plot
  p <- plot_ly(
    x = colnames(corr_matrix),
    y = rownames(corr_matrix),
    z = corr_matrix,
    type = "heatmap",
    colorscale = "RdBu",
    reversescale = TRUE,  # blue negative, red positive
    zmin = -1, 
    zmax = 1, 
    zmid = 0,
    showscale = TRUE,
    hovertemplate = "X %{x}<br>Y %{y}<br>r = %{z:.2f}<extra></extra>"
  )
  
  p <- p %>% layout(
    title = title,
    xaxis = list(title = x_label),
    yaxis = list(title = y_label)
  )
  
  # Add annotations
  if (show_annotations) {
    annotations <- list()
    for (i in seq_len(nrow(corr_matrix))) {
      for (j in seq_len(ncol(corr_matrix))) {
        val <- corr_matrix[i, j]
        annotations[[length(annotations) + 1]] <- list(
          x = colnames(corr_matrix)[j],
          y = rownames(corr_matrix)[i],
          text = sprintf("%.2f", val),
          xref = "x1", 
          yref = "y1",
          showarrow = FALSE,
          font = list(size = 11, 
                      color = ifelse(abs(val) >= 0.5, "white", "black"))
        )
      }
    }
    p <- p %>% layout(annotations = annotations)
  }
  
  return(p)
}
