% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/C00-weighted-km.R
\name{weightedKM}
\alias{weightedKM}
\title{Weighted Kaplan-Meier Estimation with Propensity Score Weights}
\usage{
weightedKM(
  data,
  treatment_var,
  time_var,
  event_var,
  ps_formula = NULL,
  weight_method = "IPW",
  att_group = NULL,
  trim = FALSE,
  delta = NULL,
  ps_control = list()
)
}
\arguments{
\item{data}{Data frame containing treatment, survival outcome, and covariates.}

\item{treatment_var}{Character string specifying the name of the treatment variable.}

\item{time_var}{Character string specifying the time-to-event variable name.}

\item{event_var}{Character string specifying the event indicator variable name.
Should be coded as 1=event, 0=censored.}

\item{ps_formula}{Formula for propensity score model: \code{treatment ~ covariates}.
Required unless \code{weight_method = "none"}.}

\item{weight_method}{Weighting method: "IPW" (inverse probability weighting),
"OW" (overlap weighting), "ATT" (average treatment effect on the treated),
or "none" (unweighted). Default "IPW".}

\item{att_group}{Target group for ATT. Required if \code{weight_method = "ATT"}.}

\item{trim}{Logical. To perform symmetric propensity score trimming? Default FALSE.
If TRUE, symmetric trimming is applied (Crump extension for multiple treatments).
See \code{\link{estimate_weights}} for trimming details. Ignored if
\code{weight_method = "none"} or \code{weight_method = "OW"}. Asymmetric
trimming is no longer supported due to poor statistical performance.}

\item{delta}{Threshold for symmetric trimming in \eqn{(0, 1/J]}, where \eqn{J} is the number
of treatment levels. Default NULL uses recommended values: 0.1 for binary
treatment, 0.067 for 3 groups, \eqn{1/(2J)} for \eqn{J \ge 4} (Yoshida et al., 2019).
Used only if \code{trim = TRUE}.}

\item{ps_control}{Control parameters for propensity score model. Default \code{list()}.
Ignored if \code{weight_method = "none"}.}
}
\value{
Object of class "weightedKM" containing:
  \item{eval_times}{Numeric vector of all unique event times.}
  \item{surv_estimates}{Matrix [n_times x n_groups] of survival probability estimates.}
  \item{surv_var}{Matrix [n_times x n_groups] of variances.}
  \item{n_risk}{Matrix [n_times x n_groups] of weighted number at risk.}
  \item{n_event}{Matrix [n_times x n_groups] of weighted number of events.}
  \item{n_acc_event}{Matrix [n_times x n_groups] of cumulative weighted events up to each time.}
  \item{treatment_levels}{Sorted unique treatment values.}
  \item{weight_method}{Weighting method used.}
  \item{att_group}{Target group for ATT (NULL if not applicable).}
  \item{trim}{Logical indicating whether trimming was applied.}
  \item{delta}{Symmetric trimming threshold (NULL if trim = FALSE).}
  \item{n}{Number of complete cases used in analysis.}
  \item{ps_result}{Propensity score estimation results (NULL if \code{weight_method = "none"}).}
  \item{weight_result}{Weight estimation results (NULL if \code{weight_method = "none"}).}
  \item{weights}{Vector of weights used in estimation (all 1s if \code{weight_method = "none"}).}
}
\description{
Computes weighted Kaplan-Meier survival or cumulative incidence curves using
propensity score weights. Supports multiple treatment groups with various
weighting schemes (IPW, OW, or ATT) and optional trimming. Special case
\code{weight_method = "none"} provides classical (unweighted) Kaplan-Meier.
}
\details{
**Weighting Methods:**

The \code{weight_method} parameter specifies the target population for causal
inference:

\itemize{
  \item \strong{IPW (Inverse Probability Weighting)}: Observations are weighted
    by the inverse probability of their observed treatment, \eqn{w_i = 1/e_j(X_i)}
    where j is the observed treatment group. Inference targets the combined
    population.

  \item \strong{OW (Overlap Weighting)}: Observations are weighted by overlap
    weights, which extends to multiple treatment groups (Li et al., 2018; 
    Li and Li, 2019). Inference targets the population at clinical equipoise 
    (overlap population).

  \item \strong{ATT (Average Treatment Effect on the Treated)}: IPW weights
    tilted toward a specified target group. Observations in the target group
    receive weight 1, others receive \eqn{w_i = e_{\text{target}}(X_i) / e_j(X_i)}.
    Inference targets the specified treatment group population.

  \item \strong{none}: No weighting applied (all weights = 1). Produces classical
    unweighted Kaplan-Meier estimates stratified by treatment.
}
}
\examples{
\donttest{
# Example 1: Classical (unweighted) Kaplan-Meier for binary treatment
data(simdata_bin)
result1 <- weightedKM(
  data = simdata_bin,
  treatment_var = "Z",
  time_var = "time",
  event_var = "event",
  weight_method = "none"
)
plot(result1)

# Example 2: Overlap-weighted Kaplan-Meier
result2 <- weightedKM(
  data = simdata_bin,
  treatment_var = "Z",
  ps_formula = Z ~ X1 + X2 + X3 + B1 + B2,
  time_var = "time",
  event_var = "event",
  weight_method = "OW"
)
summary(result2)

# Example 3: IPW-weighted Kaplan-Meier for multiple treatments
data(simdata_multi)
result3 <- weightedKM(
  data = simdata_multi,
  treatment_var = "Z",
  ps_formula = Z ~ X1 + X2 + X3 + B1 + B2,
  time_var = "time",
  event_var = "event",
  weight_method = "IPW"
)
plot(result3)

# Example 4: ATT with symmetric trimming
result4 <- weightedKM(
  data = simdata_multi,
  treatment_var = "Z",
  ps_formula = Z ~ X1 + X2 + X3 + B1 + B2,
  time_var = "time",
  event_var = "event",
  weight_method = "ATT",
  att_group = "A",
  trim = TRUE,
  delta = 0.1
)
summary(result4)
}

}
\references{
Li, F., Morgan, K. L., & Zaslavsky, A. M. (2018). Balancing covariates via
propensity score weighting. \emph{Journal of the American Statistical Association},
113(521), 390-400.

Li, F., & Li, F. (2019). Propensity score weighting for causal inference with
multiple treatments. \emph{The Annals of Applied Statistics}, 13(4), 2389-2415.

Yoshida, K., et al. (2019). Multinomial extension of propensity score trimming
methods: A simulation study. \emph{American Journal of Epidemiology}, 188(3),
609-616.
}
