% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/U02-propensity-score.R
\name{estimate_ps}
\alias{estimate_ps}
\title{Propensity Score Estimation for PSsurvival Package}
\usage{
estimate_ps(data, treatment_var, ps_formula, ps_control = list())
}
\arguments{
\item{data}{A data.frame containing the analysis data (typically the cleaned
data with complete cases).}

\item{treatment_var}{A character string specifying the name of the treatment
variable in \code{data}. Can be numeric, character, or factor with any
coding (e.g., 0/1, 1/2, "Control"/"Treated"). Function assumes treatment
has been validated for 2 or more levels.}

\item{ps_formula}{A formula object for the propensity score model, of the
form \code{treatment ~ covariates}.}

\item{ps_control}{An optional list of control parameters to pass to the
model fitting function (\code{glm} for binary treatment or
\code{nnet::multinom} for multiple treatments). Default is an empty list.}
}
\value{
A list with the following components:
  \item{ps_model}{The fitted propensity score model object (class \code{glm}
    for binary treatment or \code{multinom} for multiple treatments).}
  \item{ps}{A numeric vector of propensity scores representing the probability
    of receiving the actual treatment each individual received. Length equals
    the number of rows in \code{data}.}
  \item{ps_matrix}{A numeric matrix of dimension n × K where n is the number
    of observations and K is the number of treatment levels. Each row contains
    the predicted probabilities for all treatment levels. Column names correspond
    to treatment levels.}
  \item{n_levels}{An integer indicating the number of treatment levels.}
  \item{treatment_levels}{A vector of unique treatment values sorted by
    \code{sort()}: numerically for numeric, alphabetically for character,
    by factor level order for factor.}
}
\description{
Functions for estimating propensity scores for binary and multiple treatment groups.
Estimate Propensity Scores

Fits a propensity score model and extracts propensity scores for binary or
multiple treatment groups. For binary treatments, uses binomial logistic
regression. For multiple treatments (>2 levels), uses multinomial logistic
regression to estimate generalized propensity scores.
}
\details{
\strong{Propensity Score Definition:}
Returns P(Z = observed | X) for each individual, not P(Z=1|X) for all (as in
Rosenbaum & Rubin 1983). This definition enables direct use in IPW and extends
naturally to multiple treatments.

\strong{Binary Treatments (2 levels):}
Fits binomial logistic regression via \code{glm()}. Treatment is factorized
with levels sorted by \code{sort()}: numerically for numeric, alphabetically
for character, by factor level order for factor. Returns P(Z = observed | X).

\strong{Multiple Treatments (>2 levels):}
Fits multinomial logistic regression via \code{nnet::multinom()}. Returns
P(Z = observed | X) for each individual from the generalized PS matrix.

\strong{Control Parameters (\code{ps_control}):}
\itemize{
  \item Binary: \code{glm.control()} parameters (default: \code{epsilon=1e-08, maxit=25})
  \item Multiple: \code{multinom()} parameters (default: \code{MaxNWts=10000, maxit=100, trace=FALSE})
}
}
\examples{
\donttest{
# Example 1: Binary treatment
data(simdata_bin)
ps_bin <- estimate_ps(
  data = simdata_bin,
  treatment_var = "Z",
  ps_formula = Z ~ X1 + X2 + X3 + B1 + B2
)
summary(ps_bin$ps)
table(simdata_bin$Z)

# Example 2: Multiple treatments
data(simdata_multi)
ps_multi <- estimate_ps(
  data = simdata_multi,
  treatment_var = "Z",
  ps_formula = Z ~ X1 + X2 + X3 + B1 + B2
)
head(ps_multi$ps_matrix)
}

}
