
#' @title Area Under the Disease Progress Curve (AUDPC)
#'
#' @description
#' Calculates the Area Under the Disease Progress Curve (AUDPC)
#' using the trapezoidal integration method.
#'
#' @param time A numeric vector of time points (e.g., days after sowing or inoculation).
#' Values must be in strictly increasing order.
#'
#' @param severity A numeric vector of disease severity values
#' corresponding to each time point.
#'
#' @return A numeric value representing AUDPC.
#'
#' @details
#' AUDPC is calculated as:
#' \deqn{
#' AUDPC = \sum_{i=1}^{n-1} \frac{(Y_i + Y_{i+1})}{2} (T_{i+1} - T_i)
#' }
#'
#' where \eqn{Y} is disease severity at time \eqn{T}.
#'
#' @export
#'
#' @examples {
#' time <- c(0, 7, 14, 21, 28)
#' severity <- c(5, 15, 30, 45, 60)
#' AUDPC(time, severity)
#' }
AUDPC <- function(time, severity) {

  # Check lengths
  if (length(time) != length(severity)) {
    stop("Time and severity vectors must be of equal length")
  }

  # Check ordering
  if (any(diff(time) <= 0)) {
    stop("Time values must be in strictly increasing order")
  }

  audpc <- sum(
    ((severity[-length(severity)] + severity[-1]) / 2) *
      (time[-1] - time[-length(time)])
  )

  return(audpc)
}
