# PoissonInvSGZ.R
#' Poisson Inverse Shifted Gompertz (PISG) Distribution 
#'
#' Provides density, distribution, quantile, random generation,
#' and hazard functions for the Poisson Inverse Shifted Gompertz distribution.
#'
#' The Poisson Inverse Shifted Gompertz distribution is parameterized by the parameters
#' \eqn{\alpha > 0}, \eqn{\beta > 0}, and \eqn{\lambda > 0}.
#'
#' @details
#' The Poisson Inverse Shifted Gompertz distribution has CDF:
#'
#' \deqn{
#' F(x; \alpha, \beta, \lambda) =
#' 1 - \frac{1}{{\left( {1 - {e^{ - \lambda }}} \right)}}\left[ {1 - \exp 
#' \left\{ { - \lambda \left( {1 - {e^{ - \beta /x}}} \right)\exp 
#' \left( { - \alpha {e^{ - \beta /x}}} \right)} \right\}} \right]\quad ;\;x > 0.
#' }
#'
#' where \eqn{\alpha}, \eqn{\beta}, and \eqn{\lambda} are the parameters.
#'
#' The following functions are included:
#' \itemize{
#'   \item \code{dpois.inv.sgz()} — Density function
#'   \item \code{ppois.inv.sgz()} — Distribution function
#'   \item \code{qpois.inv.sgz()} — Quantile function
#'   \item \code{rpois.inv.sgz()} — Random generation
#'   \item \code{hpois.inv.sgz()} — Hazard function
#' }
#'
#' @usage
#' dpois.inv.sgz(x, alpha, beta, lambda, log = FALSE)
#' ppois.inv.sgz(q, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
#' qpois.inv.sgz(p, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
#' rpois.inv.sgz(n, alpha, beta, lambda)
#' hpois.inv.sgz(x, alpha, beta, lambda)
#'
#' @param x,q numeric vector of quantiles (x, q)
#' @param p numeric vector of probabilities (0 < p < 1)
#' @param n number of observations (integer > 0)
#' @param alpha positive numeric parameter
#' @param beta positive numeric parameter
#' @param lambda positive numeric parameter
#' @param log logical; if TRUE, returns log-density
#' @param lower.tail logical; if TRUE (default), probabilities are 
#' \eqn{P[X \le x]} otherwise, \eqn{P[X > x]}.
#' @param log.p logical; if TRUE, probabilities are given as log(p)
#'
#' @return
#' \itemize{
#'   \item \code{dpois.inv.sgz}: numeric vector of (log-)densities
#'   \item \code{ppois.inv.sgz}: numeric vector of probabilities
#'   \item \code{qpois.inv.sgz}: numeric vector of quantiles
#'   \item \code{rpois.inv.sgz}: numeric vector of random variates
#'   \item \code{hpois.inv.sgz}: numeric vector of hazard values
#' }
#'
#' @references  
#' Sapkota, L. P., Kumar, V., Tekle, G., Alrweili, H., Mustafa, M. S., & 
#' Yusuf, M. (2025). Fitting Real Data Sets by a New Version of Gompertz 
#' Distribution. \emph{Modern Journal of Statistics}, \bold{1(1)}, 25--48.  
#' \doi{10.64389/mjs.2025.01109}
#' 
#' @examples
#' x <- seq(0.1, 10, 0.2)
#' dpois.inv.sgz(x, 2.0, 0.5, 0.2)
#' ppois.inv.sgz(x, 2.0, 0.5, 0.2)
#' qpois.inv.sgz(0.5, 2.0, 0.5, 0.2)
#' rpois.inv.sgz(10, 2.0, 0.5, 0.2)
#' hpois.inv.sgz(x, 2.0, 0.5, 0.2)
#' 
#' # Data
#' x <- fibers69  
#' # ML estimates
#' params = list(alpha=98.0893, beta=10.6326, lambda=2.1006)
#' #P–P (probability–probability) plot
#' pp.plot(x, params = params, pfun = ppois.inv.sgz, fit.line=TRUE)
#'
#' #Q-Q (quantile–quantile) plot 
#' qq.plot(x, params = params, qfun = qpois.inv.sgz, fit.line=TRUE)
#'
#' # Goodness-of-Fit(GoF) and Model Diagnostics 
#' out <- gofic(x, params = params,
#'              dfun = dpois.inv.sgz, pfun=ppois.inv.sgz, plot=FALSE)
#' print.gofic(out)
#' 
#' @name PoissonInvSGZ
#' @aliases dpois.inv.sgz ppois.inv.sgz qpois.inv.sgz rpois.inv.sgz hpois.inv.sgz
#' @export
NULL

#' @rdname PoissonInvSGZ
#' @usage NULL
#' @export  
dpois.inv.sgz <- function (x, alpha, beta, lambda, log = FALSE)
{
	if (!all(sapply(list(x, alpha, beta, lambda), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, beta, lambda) <= 0)) {
		stop("Parameters 'alpha', 'beta' and 'lambda' must be positive.")
	}

	pdf <- numeric(length(x))

	valid <- x > 0
	if (any(valid)) {	
	tmp <- 1/(1 - exp(-lambda))
	u <- exp(-beta/x[valid])
	w <- exp(-(beta/x[valid])-alpha*u)
	z <- (1-u)* exp(-alpha*u)
    gx <- beta * lambda * w * exp(-lambda*z) * (1+alpha*(1-u)) * (1/(x[valid]*x[valid]))	
    pdf[valid] <- tmp * gx
	}
    if(log)
        pdf <- log(pdf)
    return(pdf)
}

#' @rdname PoissonInvSGZ
#' @usage NULL
#' @export  
ppois.inv.sgz <- function(q, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(q, alpha, beta, lambda), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, beta, lambda) <= 0)) {
		stop("Parameters 'alpha', 'beta' and 'lambda' must be positive.")
	}

	cdf <- numeric(length(q))

	valid <- q > 0
	if (any(valid)) {
	tmp <- 1/(1 - exp(-lambda))
	z <- (1-exp(-beta/q[valid]))* exp(-alpha* exp(-beta/q[valid]))
	cdf[valid] <- 1.0 - tmp * (1-exp(-lambda * z))
	}
    if(!lower.tail)
        cdf <- 1.0 - cdf
    if(log.p)
        cdf <- log(cdf)
    return(cdf)
}

#' @rdname PoissonInvSGZ
#' @usage NULL
#' @export  
qpois.inv.sgz <- function(p, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(p, alpha, beta, lambda), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, beta, lambda) <= 0))
		stop("Parameters 'alpha', 'beta', and 'lambda' must be positive.")

	if (log.p) p <- exp(p)
	if (!lower.tail) p <- 1 - p

	if (any(p < 0 | p > 1))
    stop("Probabilities 'p' must lie in [0, 1].")

	qtl <- numeric(length(p))
	is_zero <- p == 0
	is_one  <- p == 1
	if (any(is_zero)) qtl[is_zero] <- 0
	if (any(is_one))  qtl[is_one]  <- Inf
 
	to_solve <- which(!(is_zero | is_one))
	if (length(to_solve) > 0) {

	root_fun <- function(x, wi) {
	-log(1 - exp(-beta/x)) + alpha * exp(-beta/x) + log(-(1/lambda) * log(1-(1-exp(-lambda))*(1-wi)))   
	}
	for (i in to_solve) {
      wi <- p[i]
      lower <- 1e-4
      upper <- 1e4
      f_low <- root_fun(lower, wi)
      f_up  <- root_fun(upper, wi)

      expand_counter <- 0
      
	while (is.na(f_low) || is.na(f_up) || sign(f_low) == sign(f_up)) {
        upper <- upper * 10
        f_up <- root_fun(upper, wi)
        expand_counter <- expand_counter + 1
        if (expand_counter > 10) break
      }

      sol  <-  tryCatch(
        uniroot(function(x) root_fun(x, wi), c(lower, upper), tol = 1e-5),
        error = function(e) {
          stop(sprintf("Failed to find quantile for p = %g. Try different bracket or check parameters.", wi))
        }
      )
      qtl[i] <- sol$root
    }
  }
  return(qtl)
} 
#' @rdname PoissonInvSGZ
#' @usage NULL
#' @export 
rpois.inv.sgz <- function(n, alpha, beta, lambda)
{
	if (!is.numeric(n) || length(n) != 1L || n <= 0 || n %% 1 != 0)
		stop("'n' must be a positive integer.") 
	if (!all(sapply(list(alpha, beta, lambda), is.numeric)))
		stop("'alpha', and 'lambda' must be numeric.")
	if (any(c(alpha, beta, lambda) <= 0))
		stop("Invalid parameters: alpha, 'beta', and lambda must be positive.")
  
    u <- runif(n)
    q <- qpois.inv.sgz(u, alpha, beta, lambda)
    return(q)
}

#' @rdname PoissonInvSGZ
#' @usage NULL
#' @export 
hpois.inv.sgz <- function (x, alpha, beta, lambda)
{
	if (!all(sapply(list(x, alpha, beta, lambda), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, beta, lambda, x) <= 0)) 
		stop("All parameters and 'x' must be positive.")
     
    nume <- dpois.inv.sgz(x, alpha, beta, lambda, log=FALSE)
    surv <- 1.0 - ppois.inv.sgz(x, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
    hrf <- ifelse(surv > .Machine$double.eps, nume / surv, Inf)
    return(hrf)    
} 

