#' Forecast the GM (1, 1) model for multivariate time series data
#'
#' @description The multigreyforecast function computes the h-step ahead forecast values for each of the variables in the multivariate time series data.
#'
#' @param data Input multivariate time series data.
#' @param h The forecast horizon.
#' @return A h-step ahead forecast values
#' \item{forecast}{h-step ahead forecast values corresponding to each of the variables in the multivariate time series data.}
#'
#' @details This function returns the h-step ahead forecasted values of the fitted GM (1, 1) model for each of the variables
#'          in the multivariate time series data.
#'
#' @references
#' Akay, D. and Atak, M. (2007). Grey prediction with rolling mechanism for electricity demand forecasting of Turkey.
#' \emph{Energy}, 32(9), 1670-1675.<DOI:10.1016/j.energy.2006.11.014>
#'
#' Deng, J. (1989). Introduction to grey system theory. \emph{The Journal of Grey System}, 1(1), 1-24.
#'
#' Hsu, L.C. and Wang, C.H. (2007). Forecasting the output of integrated circuit industry using a grey model improved by
#' Bayesian analysis. \emph{Technological Forecasting and Social Change}, 74(6), 843-853.<DOI:10.1016/j.techfore.2006.02.005>
#'
#' @examples
#' # Example data
#' xt <- c(640, 684, 713, 745, 809, 811, 883, 893, 904, 935, 1044, 1069)
#' yt <- c(50, 64, 93, 113, 131, 152, 164, 201, 224, 268, 286, 290)
#' zt  <- c(550,504,493,413,431,352,364,301,294,268,286,230)
#' data <- cbind(xt, yt, zt)
#'
#' # Apply the multigreyfit function
#' multigreyforecast(data, h=3)
#' @export
#' @importFrom zoo rollapply
#' @importFrom stats lm

multigreyforecast=function(data,h)
{
  datat=t(data)
  d=dim(datat)[1]
  n=dim(datat)[2]
  #library(zoo)
  xdatat=matrix(0,d,n)
  z=matrix(0,d,n-1)
  x=matrix(0,d,n-1)
  p=matrix(0,d,2)
  forecast=matrix(0,d,h)

  for (s in 1:d) {
    xdatat[s,]=as.numeric(datat[s,])
    z[s,]=zoo::rollapply(cumsum(xdatat[s,]),2,mean)
    x[s,]=xdatat[s,-1]
    p[s,]=(lm(x[s,]~z[s,]))$coefficients

    for (j in 1:h) {
      forecast[s,j]=(xdatat[s,1]-(p[s,1]/(-p[s,2])))*exp(p[s,2]*(n+j-1))*(1-exp(-p[s,2]))
    }
  }
  output <- list("forecast"=t(forecast))
  return(output)
}

