
mean_mat <- \(n, simple = FALSE) {
    if (simple) diag(n) else matrix(1 / n, 1, n)
}

centering_mat <- \(n) {
    diag(n) - matrix(1 / n, n, n)
}

interaction_str <- \(idx, fac.name, simple = FALSE) {
    iname <- paste0(fac.name[idx], collapse = " * ")
    if (simple) {
        iname <- paste0(iname, " | ", paste0(fac.name[-idx], collapse = " * "))
    }
    iname
}

oneway_mat <- \(fac.lvls) {
    K <- prod(fac.lvls)
    L <- onewayMatrix(K)
    attr(L, "effect") <- "="
    attr(L, "scaling") <- K^2
    L
}

interaction_mat <- \(cmb, fac.lvls, simple = FALSE) {
    stopifnot(any(cmb), length(cmb) == length(fac.lvls))
    L <- matrix(1, 1, 1)
    for (i in seq_along(cmb)) {
        P <- if (cmb[i]) centering_mat(fac.lvls[i]) else mean_mat(fac.lvls[i], simple)
        L <- kronecker(L, P)
    }
    attr(L, "effect") <- interaction_str(which(cmb), names(fac.lvls), simple)
    attr(L, "scaling") <- nrow(L)
    L
}

bool_combinations <- \(n) {
    cmbs <- do.call(expand.grid, replicate(n, c(TRUE, FALSE), FALSE)) |> as.matrix() |> unname()
    rs <- rowSums(cmbs)[-nrow(cmbs)]
    ors <- order(rs, decreasing = TRUE)
    cmbs <- cmbs[ors, , drop = FALSE]
    rs <- rs[ors]
    for (s in seq_len(n)) {
        i <- which(s == rs)
        cmbs[i, ] <- cmbs[rev(i), , drop = FALSE]
    }
    cmbs
}

interaction_contrast_mat <- \(fac.lvls) {
    if (length(fac.lvls) == 1) {
        oneway_mat(fac.lvls)
    } else {
        cmbs <- bool_combinations(length(fac.lvls))
        lapply(seq_len(nrow(cmbs)), \(i) interaction_mat(cmbs[i, ], fac.lvls))
    }
}

simple_interaction_contrast_mat <- \(fac.lvls) {
    cmbs <- bool_combinations(length(fac.lvls))[-1, , drop = FALSE]
    lapply(seq_len(nrow(cmbs)), \(i) interaction_mat(cmbs[i, ], fac.lvls, TRUE))
}

get_Lmat0 <- \(H0, fac.lvls, m) {
    if (is.matrix(H0)) {
        L <- H0
        stopifnot(
            is_num_mat(L),
            all.equal(rowSums(L), rep(0, nrow(L)), check.names = FALSE)
        )
        if (is.null(attr(L, "effect"))) {
            attr(L, "effect") <- paste0("effect", m)
        } else {
            stopifnot(is_str_scalar(attr(L, "effect")))
        }
        if (is.null(attr(L, "scaling"))) {
            attr(L, "scaling") <- nrow(L)
        } else {
            s <- attr(L, "scaling")
            stopifnot(
                is_num_scalar(s),
                s > 0
            )
        }
        L
    } else if (is.list(H0)) {
        stopifnot(length(H0) == 2)
        idx <- H0[[2]]
        H0 <- H0[[1]]
        stopifnot(
            is_str_scalar(H0),
            H0 %in% c("*", "|"),
            is_idx(idx)
        )
        cmb <- rep(FALSE, length(fac.lvls))
        cmb[idx] <- TRUE
        interaction_mat(cmb, fac.lvls, H0 == "|")
    } else {
        stopifnot(is_str_scalar(H0))
        switch(H0,
            "*" = interaction_contrast_mat(fac.lvls),
            "|" = simple_interaction_contrast_mat(fac.lvls),
            "=" = oneway_mat(fac.lvls),
            stop(paste0("Unknown null hypothesis '", H0, "'."))
        )
    }
}

get_Lmat <- \(H0, fac.lvls) {
    stopifnot(any(fac.lvls > 1))
    if (!is.list(H0) || (length(H0) == 2 && any(!is.character(H0[[2]])))) {
        H0 <- list(H0)
    }
    Ls <- lapply(seq_along(H0), \(m) get_Lmat0(H0[[m]], fac.lvls, m)) |> rrapply(how = "flatten")
    L <- do.call(rbind, Ls)
    attr(L, "effect") <- sapply(Ls, attr, "effect")
    attr(L, "size") <- sapply(Ls, nrow)
    attr(L, "scaling") <- sapply(Ls, attr, "scaling")
    L
}
