#' Update equation for lambda
#'
#' @param lambda A scalar defining the parameter for the Truncate Poisson distribution
#'    that controls the number of change points (or its initial values)
#' @param kstar A scalar with the number maximum of change points in all clusters
#' @param N A scalar representing the number of data sequences
#' @param al The hyperparameter value for the shape parameter in the gamma prior for lambda
#' @param bl The hyperparameter value for the scale parameter in the gamma prior for lambda
#' @param a The hyperparameter value for the shape parameter in the gamma prior for alpha0
#' @param b The hyperparameter value for the scale parameter in the gamma prior for alpha0
#' @param K A vector containing the number of change points for each cluster (or its initial values)
#' @param cluster A vector containing the cluster assignments for the data sequences (or its initial values)
#'
#' @returns A numerical value corresponding to a sample from the posterior of the parameter lambda
#'
#' @note
#' This function is called within the Gibbs sampler, but it can also be called separately.
#'
#' @seealso [gibbs_alg()]
#'
#' @examples
#' update_lambda(a = 4, b = 2, kstar = 2, lambda = 2, cluster = c(1,1,2,1,2),
#'  al = 2, bl = 1000, K = c(2,2), N = 5)
#'
#' @export
#'
update_lambda <- function(a = 4, b = 2, kstar, lambda, cluster, al, bl, K, N){

  lambda_prop <- stats::rgamma(1, a, b)

  log.r <- log(full_cond(kstar=kstar, lambda=lambda_prop, cluster=cluster, al=al, bl=bl, K=K, N=N)) - log(full_cond(kstar=kstar, lambda=lambda, cluster=cluster, al=al, bl=bl, K=K, N=N)) + stats::dgamma(lambda, a, b, log = TRUE) - stats::dgamma(lambda_prop, a, b, log = TRUE)

  if(log(stats::runif(1)) < log.r){lambda <- lambda_prop}

  return(lambda)
}

