\name{ffnet.terasvirta.test}
\title{Teraesvirta Neural Network Test for Nonlinearity and Model Selection}
\usage{
ffnet.terasvirta.test (x, y, nhid = 0, alpha = 0.9, scale = TRUE,
                       errfunc, outtype, init = c("null", "train"),
                       trace = c("PRINT","NO","PLOT"), ...)
ffnet.terasvirta.test (y, x, lag = 1, lagx = 1, nhid = 0, alpha = 0.9, 
                       scale = TRUE, errfunc, init = c("null", "train"),
                       trace = c("PRINT","NO","PLOT"), ...)
ffnet.terasvirta.test (formula, data = NULL, errfunc, outtype, ...)
}
\alias{ffnet.terasvirta.test}
\alias{ffnet.terasvirta.test.default}
\alias{ffnet.terasvirta.test.ts}
\alias{ffnet.terasvirta.test.formula}
\arguments{
  \item{x}{for \code{ffnet.terasvirta.test.default} a matrix, vector, or
    data frame of (input) examples. For \code{ffnet.terasvirta.test.ts}
    the time series of exogenous variables. Only \code{x(t-1)} is
    included in the model to explain \code{y(t)}.} 
  \item{y}{for \code{ffnet.terasvirta.test.default} a matrix, vector, or data frame of
    target examples. For \code{ffnet.terasvirta.test.ts} the time series
    of endogenous variables.}
  \item{nhid}{the number of hidden units for the null model.}
  \item{alpha}{the proportion of the total variance that is explained by
    the most important principal components.}
  \item{scale}{a logical indicating whether the data should be scaled 
    before computing the test statistic. The default arguments to
    \code{\link{scale}} are used.}
  \item{errfunc}{a string which specifies the type of objective
    function to be used for the test. See also \code{\link{ffnet}} for a
    description.}
  \item{outtype}{specifies the type of output unit activation
    functions to be used. See also \code{\link{ffnet}} for a
    description.} 
  \item{init}{a string indicating how the alternative model is
    initialized for training. It must be one of \code{"null"} (the default) or
    \code{"train"}. \code{"null"} means that the alternative model is
    initialized with the null (sub)model. \code{"train"} means that the
    alternative model is initialized by means of a preliminary training.}
  \item{trace}{a string indicating the type of optimizer output. It must
    be one of \code{"PRINT"} (the default), \code{"PLOT"}, or
    \code{"NO"}. See also \code{\link{ffnet}} for a description.}
  \item{\dots}{additional arguments to \code{\link{ffnet}} for
    preliminary training of the null model and of the alternative model
    if \code{init} is set to \code{"train"}.}
  \item{lag}{an integer indicating the time series model order in terms of
    lags.}
  \item{lagx}{an integer specifying the number of lagged \code{x} that
    are included in the model.}
  \item{formula}{a symbolic description of the null model to be tested. Note,
    that an intercept is always included, whether given in the formula
    or not.}
  \item{data}{an optional data frame containing the variables in
    the model. By default the variables are taken
    from the environment which \code{ffnet.terasvirta.test} is called
    from.}
  \item{\dots}{additional arguments to
    \code{ffnet.terasvirta.test.default}.} 
}
\description{
  These generic procedures compute Teraesvirta's neural network test for
  neglected nonlinearity, either for a regression-type model of the form

  \code{y~x},   or    \code{formula},

  or a time series model of order \code{lag} for \code{y} with exogenous
  variables \code{x} of the form
  
  \code{y[t] = NN (y[t-1], \dots, y[t-lag], x[t-1], \dots, x[t-lagx]) +
    e[t]}.

  The null is the hypotheses that the model with \code{nhid} hidden
  units is correctly specified in terms of not neglecting any
  nonlinearity in "mean".
}
\details{
  The test procedure is implemented as a likelihood-ratio test with
  the null model containing \code{nhid} hidden units and the alternative
  model containing \code{nhid+1} hidden units. To solve the
  identification problem, a Taylor series expansion of the activation
  function of the \code{nhid+first} hidden unit is applied. To improve
  the power of the test, the terms of the expansion are replaced by their
  most important principal components. The number of principal
  components is chosen as the minimum number of components that explain
  at least a proportion \code{alpha} of the total variance. The test
  statistic is Chi-Square distributed with the number of selected principal
  components times the dimension of the output vector degrees of freedom.
  
  Note, that these tests can also be used for bottom up model selection
  by applying a sequence of likelihood-ratio tests.
  
  Missing values are not allowed.
}
\value{
  A list with class \code{"htest"} containing the following components:
  \item{statistic}{the value of the test statistic.}
  \item{p.value}{the p-value of the test.}
  \item{method}{a character string indicating what type of test was
    performed.}
  \item{parameter}{a list containing the test parameters degrees of freedom
    \code{"df"}, null number of hidden units 
    \code{"nhid"}, and for time series models the null model order
    \code{"lag"} and eventually \code{"lagx"}.}
  \item{data.name}{a character string giving the name of the data.}
  \item{arguments}{a list containing the additional arguments proportion
    of total explained variation \code{"alpha"} and scaled data
    \code{"scale"}. Furthermore, the list contains the error function value of
    the fitted null and fitted alternative model \code{"null fitted value"} and
    \code{"alternative fitted value"}, respectively.}
}
\references{
  T. Teraesvirta, C. F. Lin, and C. W. J. Granger (1993): Power of the
  Neural Network Linearity Test. \emph{Journal of Time Series Analysis}
  \bold{14}, 209-220.

  U. Anders and O. Korn (1999): Model Selection in Neural
  Networks. \emph{Neural Networks} \bold{12}, 309-323.
}
\author{A. Trapletti}
\seealso{
  \code{\link{ffnet}}
}
\examples{
n <- 1000

x <- runif (1000, -1, 1)  # Non-linear in "mean" regression 
y <- x^2 - x^3 + 0.1*rnorm(x)

ffnet.terasvirta.test (y~x)  # Linear or nonlinear model?

# one or two hidden units?
# preliminary estimation to get rid of local minima
# set itmax to some bigger value than demonstrated here!
ffnet.terasvirta.test (y~x, nhid=1, optim="NRSANN", temp=1, itmax=1000)  

x[1] <- 0.0
for (i in (2:n))  # Generate time series which is nonlinear in "mean"
{
  x[i] <- 0.4*x[i-1] + tanh(x[i-1]) + rnorm (1, sd=0.5)
}
x <- as.ts(x)
plot (x)
ffnet.terasvirta.test (x, lag=1)

ffnet.terasvirta.test (x, lag=1, nhid=1, optim="NRSANN", temp=1, itmax=1000) 
}
\keyword{ts}




